
library(dplyr)

amphiData <- read.csv(file = "./Data/Amphibians_in_trade.csv",
                      stringsAsFactors = FALSE)

lemisData <- read.csv(file = "./Data/LEMIS Data AmphiNames.csv",
                      stringsAsFactors = FALSE)

temporalDataOnline <- read.csv(file = "./Data/Temporal Online Data.csv",
                               stringsAsFactors = FALSE)

firstYearTradedOnline <- temporalDataOnline %>% 
  group_by(sp) %>% 
  summarise(tradeYear = min(year)) %>% 
  rename("amphiName" = sp)

firstYearTradedLemis <- lemisData %>% 
  group_by(amphiName) %>% 
  summarise(tradeYear = min(shipment_year))

amphiData$tradeYear <- apply(amphiData, 1, function(x){
  if(!x["amphiName"] %in% unique(c(firstYearTradedLemis$amphiName,
                                   firstYearTradedOnline$amphiName))){
    return(NA)
  } else {
    return(min(
      c(
        firstYearTradedOnline[which(firstYearTradedOnline$amphiName == x["amphiName"]),]$tradeYear,
        firstYearTradedLemis[which(firstYearTradedLemis$amphiName == x["amphiName"]),]$tradeYear
      )
    ))
  }
})
# where they were detected in online trade but not the temp sample we know they
# appear at earliest in 2020
amphiData$tradeYear[is.na(amphiData$tradeYear) & amphiData$onlineEither] <- 2020

library(ggplot2)
library(scico)
library(stringr)
library(ggtext)
library(glue)

cols <- scico(2, palette = "roma")

label <- amphiData %>% 
  mutate(tradeLag = tradeYear - descYear) %>% 
  filter(descYear > 1999, tradeYear > 1999) %>% 
  mutate(snapshotOnly = ifelse(tradeYear == 2020, "Snapshot\nonly", "Temporal\ndata"),
         snapshotOnly = factor(snapshotOnly, levels = c("Temporal\ndata", "Snapshot\nonly"))) %>% 
  filter(snapshotOnly == "Temporal\ndata") %>% 
  summarise(mean = round(digits = 2,mean(tradeLag)),
            se = round(digits = 2,sqrt(var(tradeLag)/length(tradeLag))))

lagPlot <- amphiData %>% 
  mutate(tradeLag = tradeYear - descYear,
         colour = case_when(
           iucn == "Least Concern (LC)" ~ "#309706",
           iucn == "Near Threatened (NT)" ~ "#A8DB06", 
           iucn == "Vulnerable (VU)" ~ "#F5D800",
           iucn == "Endangered (EN)" ~ "#DC7000", 
           iucn == "Critically Endangered (CR)" ~ "#CB1500", 
           iucn == "Extinct in the Wild (EW)" ~ "#701E08", 
           iucn == "Extinct (EX)" ~ "#000000",
           iucn == "Data Deficient (DD)" ~ "#717171",
           is.na(iucn) ~ "#BFBFBF"),
         iucnSimp = sub("\\).*", "", sub(".*\\(", "", iucn)), 
         iucnCol = glue::glue("<i>{amphiName}</i> <b style='color:{colour}'>({iucnSimp})</b>")
  ) %>% 
  filter(descYear > 1999, tradeYear > 1999) %>% 
  mutate(snapshotOnly = ifelse(tradeYear == 2020, "Snapshot\nonly", "Temporal\ndata"),
         snapshotOnly = factor(snapshotOnly, levels = c("Temporal\ndata", "Snapshot\nonly")),
         citesApp = ifelse(citesApp == FALSE, "", citesApp)) %>%
  ggplot(aes(fill = iucnCol)) +
  geom_segment(aes(x = descYear, xend = tradeYear,
                   y = reorder(iucnCol, -descYear),
                   yend = reorder(iucnCol, -descYear))) +
  geom_point(aes(x = descYear, y = iucnCol), colour = cols[2], size = 2,
             pch = 16) +
  geom_point(aes(x = tradeYear, y = iucnCol), colour = cols[1], size = 2,
             pch = 16) +
  geom_point(aes(x = 2021, y = iucnCol, shape = citesApp), size = 4) +
  geom_text(aes(x = 2021, y = iucnCol, label = citesApp),
            size = 3, fontface = 2,
            colour = "white", vjust = 0.5) +
  scale_shape_manual(values = c(NA, 16)) +
  facet_grid(snapshotOnly~., drop = TRUE, scales = "free_y", space = "free_y") +
  scale_x_continuous(breaks = seq(2000, 2021, 1), minor_breaks = NULL,
                     limits = c(2000, 2021),
                     labels = c(paste0("'", str_pad(seq(00, 20, 1), 2, "left", 0)),
                                "App.")) +
  theme_bw() +
  labs(x = "Year", y = "Species",
       colour = "", tag = "(LEMIS only)\n2000-2001") +
  theme(legend.position = "none",
        panel.background = element_blank(),
        panel.border = element_blank(),
        axis.line = element_line(),
        axis.title = element_text(face = 2),
        axis.title.y = element_text(angle = 0, hjust = 1,
                                    margin = margin(0,5,0,0)),
        axis.text.y = element_markdown(),
        strip.background = element_blank(),
        # strip.text.y = element_text(face = 4, hjust = 0, angle = 0, vjust = 1)
        strip.text = element_blank(),
        plot.tag.position = c(0.385, 0.005),
        plot.tag = element_text(hjust = 0.5, vjust = 0, size = 6, lineheight = 0.85))

lagHist <- amphiData %>% 
  filter(descYear > 1999, tradeYear > 1999) %>% 
  mutate(snapshotOnly = ifelse(tradeYear == 2020, "Snapshot\nonly", "Temporal\ndata"),
         snapshotOnly = factor(snapshotOnly, levels = c("Temporal\ndata", "Snapshot\nonly")),
         tradeLag = tradeYear - descYear,
         iucnSimp = factor(case_when(
           iucn %in% c(
             "Endangered (EN)",
             "Critically Endangered (CR)",
             "Vulnerable (VU)",
             "Extinct in the Wild (EW)",
             "Extinct (EX)") ~
             "Threatened<br>(<b style='color:#F5D800'>VU</b>, 
           <b style='color:#DC7000'>EN</b>,
           <b style='color:#CB1500'>CR</b>,
           <b style='color:#701E08'>EW</b>,
           <b style='color:#000000'>EX</b>)",
           iucn %in% c("Near Threatened (NT)", "Least Concern (LC)") ~
             "Not threatened<br>(<b style='color:#A8DB06'>NT</b>, 
           <b style='color:#309706'>LC</b>)",
           iucn == "Data Deficient (DD)" | is.na(iucn) ~
             "Lacking data or unevalulated<br>(<b style='color:#717171'>DD</b>, 
           <b style='color:#BFBFBF'>NA</b>)"),
           levels = c("Threatened<br>(<b style='color:#F5D800'>VU</b>, 
           <b style='color:#DC7000'>EN</b>,
           <b style='color:#CB1500'>CR</b>,
           <b style='color:#701E08'>EW</b>,
           <b style='color:#000000'>EX</b>)",
           "Not threatened<br>(<b style='color:#A8DB06'>NT</b>, 
           <b style='color:#309706'>LC</b>)",
           "Lacking data or unevalulated<br>(<b style='color:#717171'>DD</b>, 
           <b style='color:#BFBFBF'>NA</b>)"
           )
         )) %>%
  filter(snapshotOnly == "Temporal\ndata") %>% 
  ggplot() +
  stat_count(aes(x = tradeLag, fill = iucnSimp), alpha = 0.65,
             width = 0.75) +
  geom_vline(xintercept = label$mean, linetype = 2) +
  geom_text(data = label, aes(x = mean+0.25, y = 4,
                              label =  paste0("\nMean ±se\n",
                              mean, " ±", se)),
            hjust = 0, lineheight = 0.85, fontface = 4, size = 3)+
  scale_x_continuous(breaks = seq(0,20,1),
                     minor_breaks = NULL) +
  scale_y_continuous(minor_breaks = NULL) +
  scale_fill_manual(values = c("#DC7000", "#309706", "#717171")) +
  theme_bw() +
  labs(y = "Count", x = "Lag between description and appearance in the trade (years)",
       fill = "Simplified IUCN status") +
  theme(legend.position = "left",
        legend.text = element_markdown(),
        legend.title = element_text(face = 2),
        panel.background = element_blank(),
        panel.border = element_blank(),
        axis.line = element_line(),
        axis.title = element_text(face = 2),
        axis.title.y = element_text(angle = 0, hjust = 1,
                                    margin = margin(0,5,0,0)))

library(cowplot)
library(ggpubr)

plot_grid(lagPlot +
            theme(plot.margin = margin(10,5,5,25)),
          lagHist +
            theme(plot.margin = margin(5,5,5,25)),
          ncol = 1,
          axis = "lr",
          align = "v",
          rel_heights = c(2.5,1),
          labels = c("A", "C", "")
          ) +
  annotate("text", x = 0.016, y = 0.48, size = 5,
           label = "B", fontface = 2) +
  # geom_richtext(data = amphiData %>% 
  #             summarise(post1999desc = sum(descYear > 1999),
  #                       post1999trade = sum(descYear > 1999 & tradeYear > 1999, na.rm = TRUE)) %>% 
  #             mutate(per1999trade = post1999trade / post1999desc *100),
  #           aes(x = 0.18, y = 0.14,
  #               label = 
  #                 glue("<span style='font-size:18pt; color:black'>**Summary**</span><br>
  #                      **{post1999trade}** post-1999 species traded<br>
  #                      **{post1999desc}** post-1999 species descriptions<br>
  #                      **{round(per1999trade, digits = 2)}%** of post-1999 species traded")
  #               ),
  #           hjust = 0.5, fontface = 1,
  #           fill = NA, label.color = NA,
  #           lineheight = 1.6,
  #           label.padding = grid::unit(rep(0, 4), "pt")) +
  NULL

ggsave("./Figures/Desc to trade plot.png", width = 240, height = 170,
       dpi = 300, units = "mm")  
ggsave("./Figures/Desc to trade plot.pdf", width = 240, height = 170,
       units = "mm")

amphiData %>% 
  summarise(post1999desc = sum(descYear > 1999),
            post1999trade = sum(descYear > 1999 & tradeYear > 1999, na.rm = TRUE)) %>%
  mutate(per1999trade = post1999trade / post1999desc *100)

amphiData %>% 
  filter(descYear > 1999 & tradeYear > 1999) %>% 
  select(citesApp, amphiName)
